type known_values =
  | String of (string * string array)
  | Enum of (string * (string * Obj.t) array * string)
  | Int of (int * int array)
  | Float of (float * float array)
  | Bool of bool

let type_of_known_values = function
  | String _ -> "string"
  | Enum (_, _, t) -> t
  | Int _ -> "int"
  | Float _ -> "float"
  | Bool _ -> "bool"

let default_of_known_values ?(copy=false) = function
  | String (def, _a) -> if copy then Printf.sprintf "String.copy %S" def else def
  | Enum (def, _a, _t) -> def
  | Int (def, _a) -> string_of_int def
  | Float (def, _a) -> string_of_float def
  | Bool def -> string_of_bool def

let values_string_of_known_values = function
  | String (_def, a) -> a
  | Enum (_def, a, _t) -> Array.map fst a
  | Int (_def, a) -> Array.map string_of_int a
  | Float (_def, a) -> Array.map string_of_float a
  | Bool _def -> [| "true"; "false" |]

type 'a wrap = {
  name : string;
  short_description : string;
  long_description : string;
  level : int;
  value : 'a;
}

type opt = {
  known_values : known_values;
  can_be_user_specified : bool;
  immediate : bool;
}

type category = {
  sub_categories : category wrap array;
  options : opt wrap array;
}

let dup_obj_with : 'a -> int -> 'b -> 'a = fun obj i v ->
  let obj' = Obj.dup (Obj.repr obj) in
  Obj.set_field obj' i (Obj.repr v);
  Obj.obj obj'
  
let get_field obj i =
  Obj.obj (Obj.field (Obj.repr obj) i)

let rec option_of_path ~options = function
  | [ i ] -> get_field options i
  | t :: q -> option_of_path ~options:(get_field options t) q
  | [] -> raise Not_found

let value_index ~option ~option_descr =
  let find v a =
    let v = Obj.magic v in
    let i = ref 0 in
    try
      while (Array.get a !i) <> v do
        incr i;
      done;
      !i
    with Invalid_argument "index out of bounds" -> raise Not_found
  in
  match option_descr.value.known_values with
  | String (_def, a) -> find option a
  | Enum (_def, a, _) -> find option (Array.map snd a)
  | Int (_def, a) -> find option a
  | Float (_def, a) -> find option a
  | Bool _def -> if option then 1 else 0
  
let ith_option_descr ~i ~known_values =
  match known_values with
  | String (_def, a) -> Obj.repr a.(i)
  | Enum (_def, a, _) -> snd a.(i)
  | Int (_def, a) -> Obj.repr a.(i)
  | Float (_def, a) -> Obj.repr a.(i)
  | Bool _def -> Obj.repr (i <> 0)

(* TODO: validate the update path against the option description! *)
let rec update category value = function
  | [] -> assert false
  | [ i ] -> dup_obj_with category i value
  | i :: q -> dup_obj_with category i (update (get_field category i) value q)

let shared_category =
  ref { name = ""; short_description = ""; long_description = ""; level =
    0; value = { sub_categories = [| |]; options = [| |] } }

