open BrowserModel.Types

let i = ignore

let button_bar ~packing ~core =
  let button_bar = GPack.button_box `HORIZONTAL ~packing ~layout:`START () in
  GButton.button ~label:"Options" ~use_mnemonic:true ~stock:`PREFERENCES ~packing:button_bar#add ()

let shortcuts =
  let make_shortcut ~name ~shortcut =
    let open GtkSignal in
    { name; classe = `window; marshaller = marshal_unit }, shortcut
  in
  [
    make_shortcut ~name:"create-new-page" ~shortcut:"<Control>T";
    make_shortcut ~name:"close-page" ~shortcut:"<Control>W";
  ]

class page ?parent_id ?(show=true) ~id ~propagator ?uri options =
  let core = new Page.core ~state_callbacks:[propagator] (id, parent_id, options) in
  let vbox = GPack.vbox () in
  let address = GEdit.entry ~packing:(vbox#pack ~expand:false) () in
  let scroll = GBin.scrolled_window ~hpolicy:`AUTOMATIC ~vpolicy:`AUTOMATIC
    ~packing:(vbox#pack ~expand:true) () in
  let webview = GWebView.web_view ~packing:scroll#add () in
  let status = GMisc.statusbar ~packing:(vbox#pack ~expand:false) () in
  let status_context = status#new_context ~name:"status" in
  let status_messages = Stack.create () in
  let label = new LibUi.Label.label ~title:"about:blank" () in
  let () = label#text#set_width_chars 14 in
  object (self)
    val mutable foo = None
    method label = label#coerce
    method webview = webview
    method coerce = vbox#coerce
    method send = core#send
    method set_foo (bar : GWebView.web_view option) = foo <- bar
    method private close () =
      vbox#destroy (); webview#destroy (); address#destroy ();
      status#destroy ()
    method sink = function
      | `Uri uri -> webview#load_uri uri
      | `Title title -> prerr_endline title
      | `Close -> self#close ()
      | `SetOptions _options -> ()
      | `Request _ -> ()
    initializer
      core#msg_callbacks#add self#sink;
      i (webview#connect#notify_title ~callback:label#text#set_text);
      i (label#close#connect#clicked ~callback:(fun () -> core#send (`Request `Close)));
      let main_frame = new GWebFrame.web_frame webview#get_main_frame in
      i (main_frame#connect#notify_uri ~callback:address#set_text);
      i (webview#connect#create_web_view ~callback:(fun web_frame ->
        core#send (`Request `Popup);
        match foo with
        | Some wv -> wv#as_webview
        | None -> raise (Failure "moo!")
      ));
      i (webview#connect#hovering_over_link ~callback:(fun title uri ->
        if uri <> "" then
          let title = if title <> "" then title else "no link title" in
          let msg = status_context#push (Printf.sprintf "%s (%s)" uri title) in
          Stack.push msg status_messages
        else
          let () = Stack.iter status_context#remove status_messages in
          Stack.clear status_messages
      ));
      i (webview#connect#mime_type_policy_decision_requested ~callback:(
        fun web_frame request mime policy_decision ->
          try
            let slash_i = String.index mime '/' in
            let first_part = String.sub mime 0 slash_i in
            match first_part with
            | "application" ->
                Printf.eprintf "Download: %s\n%!" request#uri;
                false
            | _ -> true
          with _ ->
            Printf.eprintf "Unhandled mime type: `%s'\n%!" mime;
            false
        ));
      i (address#connect#activate ~callback:(fun () -> core#send (`Uri address#text)));
      if show then
        let uri = match uri with
        | None -> options.BrowserModel.Options.new_page_default_uri
        | Some uri -> uri
        in
        core#send (`Uri uri)
  end

class navigation ~packing ~propagator options =
  let core = new Navigation.core ~state_callbacks:[propagator] options in
  let tabs = GPack.notebook ~homogeneous_tabs:true ~tab_pos:options.BrowserModel.Options.tabs_position ~packing ~scrollable:true () in
  let propagator state = core#send (`Set state) in
  let id_gen =
    let id_gen = LablgtkReact.id_gen () in
    fun () -> id_gen ()
  in
  object (self)
    val pages : (LablgtkReact.id, page) Hashtbl.t = Hashtbl.create 100
    val mutable options = options
    initializer
      i (tabs#connect#after#switch_page ~callback:(fun _ i -> core#send (`Nth i)));
      i (tabs#connect#after#page_reordered ~callback:(fun _ i -> core#send (`MoveTo i)));
      core#msg_callbacks#add self#sink
    method new_page ?uri ?parent_id () =
      let id = id_gen () in
      let show = parent_id = None in
      let page = new page ~id ~show ?parent_id ~propagator ?uri options.BrowserModel.Options.page in
      Hashtbl.add pages id page;
      i (LibUi.Notebook.insert ~notebook:tabs ~tab_label:page#label ~pos:options.BrowserModel.Options.new_tab_placement ~page:page#coerce);
      match parent_id with
      | None -> ()
      | Some parent_id ->
          let parent_page = Hashtbl.find pages parent_id in
          parent_page#set_foo (Some page#webview)
    method tabs = tabs
    method send = core#send
    method sink = function
      | `Close (id, activate)->
          (* XXX: to be done after removal because the index will change and has
           * been computed post-removal in the model. *)
          Gaux.may ~f:tabs#goto_page activate;
          let page = Hashtbl.find pages id in
          Hashtbl.remove pages id;
          page#send `Close
      | `Nth i ->
          if i <> tabs#current_page then
            tabs#goto_page i
      | `MoveTo i ->
          if i <> tabs#current_page then
            tabs#reorder_child (tabs#get_nth_page tabs#current_page) i
      | `SetOptions new_options ->
          options <- new_options;
          tabs#set_tab_pos options.BrowserModel.Options.tabs_position;
          Hashtbl.iter (fun _id page -> page#send (`SetOptions new_options.BrowserModel.Options.page)) pages
      | `None -> ()
      | `Popup parent_id -> self#new_page ~parent_id ()
      | `Add id -> ()
  end

class browser ~options =
  let core = new Browser.core options in
  let window = GWindow.window ~width:800 ~height:600 ~allow_shrink:true () in
  let vbox = GPack.vbox ~packing:window#add () in
  let show_options = button_bar ~core ~packing:vbox#pack in
  let propagator state = core#send (`Set state) in
  let navigation = new navigation ~propagator ~packing:(vbox#pack ~expand:true)
  options.BrowserModel.Options.navigation in
  object (self)
    val core = core
    val vbox = vbox
    val button_bar = button_bar
    val mutable options = options
    method new_page ?uri () = navigation#new_page ?uri ()
    method window = window
    method private sink = function
      | `SetOptions new_options ->
          options <- new_options;
          navigation#send (`SetOptions options.BrowserModel.Options.navigation)
      | `None -> ()
    initializer
      i (window#connect#destroy ~callback:GMain.Main.quit);
      let options_send options = core#send (`SetOptions options) in
      i (show_options#connect#clicked ~callback:(fun () -> OptionsWindow.build ~send:options_send ~options_descr:BrowserModel.OptionsDescr.browser ~options));
      core#msg_callbacks#add self#sink;
      LibUi.Shortcuts.create_shortcuts shortcuts window#as_window;
      StdLabels.List.iter shortcuts ~f:(fun (sgn, _shortcut) ->
        let open GtkSignal in
        let f ~sgn callback = i (connect ~sgn ~callback window#as_window) in
        match sgn.name with
        | "create-new-page" -> f ~sgn (fun () -> navigation#new_page ())
        | "close-page" -> f ~sgn (fun () -> navigation#send `CloseCurrent)
        | _ -> ()
      )
  end

