module Page0 = struct
  type options = Options.page

  type t = {
    uri : string;
    title : string;
    history : string list;
    options : options;
    id : LablgtkReact.id;
    parent_id : LablgtkReact.id option;
  }

  type init = (LablgtkReact.id * LablgtkReact.id option * options)

  type action_common = [
    | `Uri of string
    | `Title of string
    | `SetOptions of options
    | `Close
  ]

  type action_functional = [
    action_common
    (* These requests are from the view itself: from the UI. *)
    | `Request of [ `Close | `Popup ]
  ]

  type action_ui = [
    action_common
    | `Request of [ `Add | `Close | `Popup ]
  ]

  let create (id, parent_id, options) =
    {
      uri = "about:blank";
      title = "(empty)";
      history = [];
      options = options;
      id = id;
      parent_id = parent_id;
    }, `Request `Add

  let rewrite ~t uri = (* XXX: ***BAD*** but should be good enough for now *)
    if try (String.index uri ':' < 8) with Not_found _ -> false then
      uri
    else
      String.concat "://" [ t.options.Options.default_uri_scheme; uri ]

  let on_action_func t (msg : action_functional) = match msg with
    | `Uri uri as msg ->
        let uri = rewrite ~t uri in
        { t with uri; history = uri :: t.history }, msg
    | `Title title as msg -> { t with title }, msg
    | `SetOptions options as msg -> { t with options }, msg
    | `Request `Close as msg -> t, msg
    | `Request `Popup as msg -> t, msg
    | `Close as msg -> t, msg
end

module Navigation0 = struct
  module Z = Zipper
  type t = {
    z : Page0.t Z.t;
    options : Options.navigation;
  }

  type init = Options.navigation

  type action_common = [
    | `Nth of int
    | `MoveTo of int
    | `SetOptions of Options.navigation
  ]

  type action_functional = [
    action_common
    | `CloseCurrent
    | `Set of Page0.t * Page0.action_ui
  ]

  type action_ui = [
    action_common
    | `Close of (LablgtkReact.id * int option)
    | `Popup of LablgtkReact.id
    | `Add of LablgtkReact.id
    | `None
  ]

  let create options =
    { z = Z.empty; options = options }, `None

  let close_current n =
    let z = Z.remove n.options.Options.post_close_activate n.z in
    { n with z }, `Close ((Z.get n.z).Page0.id, Some (Z.pos z))

  let on_action_func n (action : action_functional) = match action with
  | `Nth i ->
      if i = Z.pos n.z then
        n, `None
      else
        { n with z = Z.nth n.z i }, `Nth i
  | `MoveTo i ->
      let cur = Z.pos n.z in
      if i = cur then
        n, `None
      else
        { n with z = Z.move n.z cur i }, `MoveTo i
  | `CloseCurrent ->
      close_current n
  | `Set (e, msg) ->
      let find () = Z.find (fun z -> (Z.get z).Page0.id = e.Page0.id) n.z in
      let n_of z = { n with z = Z.nth z (Z.pos n.z)  } in
      (match msg with
      | `Request `Add ->
          { n with z = Z.add n.options.Options.new_tab_placement n.z e }, `Add e.Page0.id
      | `Request `Popup -> n_of (Z.set (find ()) e), `Popup e.Page0.id
      | `Request `Close when Page0.(e.id = (Z.get n.z).id) -> close_current n
      | `Request `Close -> n_of (Z.remove `After (find ())), `Close (e.Page0.id, None)
      | `Close -> n, `None
      | _ -> n_of (Z.set (find ()) e), `None)
  | `SetOptions options as msg -> { n with options }, msg
end

module Browser0 = struct
  type options = Options.browser

  type t = {
    options : options;
    navigation : Navigation0.t;
  }

  type init = options

  type action_common = [ `SetOptions of options ]

  type action_functional = [
    action_common
    | `Set of (Navigation0.t * Navigation0.action_ui)
  ]

  type action_ui = [ action_common | `None ]

  let create options =
    let navigation, _msg = Navigation0.create options.Options.navigation in
    { options; navigation }, `None

  let save_options options =
    try
      let oc = open_out_bin "caravel.conf" in
      Marshal.to_channel oc options [];
      close_out oc
    with _ -> 
      ()

  let on_action_func t = function
    | `Set (navigation, _msg) -> { t with navigation }, `None
    | `SetOptions options as msg ->
        save_options options;
        { t with options }, msg
end

